﻿/*
    SocialControls - http://fabi.me/dev/socialcontrols/
    Copyright (C) 2011 Fabian Schlieper

    
    This library is released under the following license:
    
    Creative Commons Attribution-NonCommercial-ShareAlike 3.0 Unported
    http://creativecommons.org/licenses/by-nc-sa/3.0/
    
    Author: Fabian Schlieper <fabian@fabi.me>
*/

using System;
using System.ComponentModel;
using System.Drawing;
using System.Text;
using System.Windows.Forms;
using System.Globalization;

namespace SocialControls
{
    [ToolboxBitmapAttribute(typeof(FacebookLikeButton), "Facebook16.bmp")]
    public partial class FacebookLikeButton : UserControl
    {
        public enum Verbs
        {
            Like, Recommend
        }

        public enum ColorSchemes
        {
            Light, Dark
        }

        string thingUrl;
        bool compact;
        Verbs verb;
        ColorSchemes colorscheme;

        private void updateBtn()
        {
            if (!this.IsHandleCreated) return;

            this.Size = compact ? new System.Drawing.Size(110, 20) : new System.Drawing.Size(85, 62);

            webBrowser.Hide();

            if (!String.IsNullOrEmpty(thingUrl))
            {
                StringBuilder sb = new StringBuilder("http://www.facebook.com/plugins/like.php?href=");
                sb.Append(Uri.EscapeDataString(thingUrl));
                sb.Append("&layout="); sb.Append(compact?"button_count":"box_count");
                sb.Append("&show_faces=false");
                sb.Append("&width="); sb.Append(this.Width);
                sb.Append("&height="); sb.Append(this.Height);
                sb.Append("&action="); sb.Append(verb.ToString().ToLower());
                sb.Append("&colorscheme="); sb.Append(colorscheme.ToString().ToLower());
                if (compact)
                    sb.Append("&button=compact");
                sb.Append("&locale="); sb.Append(CultureInfo.CurrentUICulture.IetfLanguageTag.Replace('-','_'));
                webBrowser.Navigate(sb.ToString());
            }
        }

        /// <summary>
        /// The URL of the Thing to like
        /// </summary>
        [Description("Thing URL"),
         Category("Facebook Like"),
         DefaultValue(""),
         Browsable(true),
         Bindable(true)]
        public string ThingUrl
        {
            get { return thingUrl; }
            set
            {
                if (thingUrl == value) return;
                thingUrl = value;
                updateBtn();
            }
        }

        /// <summary>
        /// Use a compact button layout
        /// </summary>
        [Description("Compact Button"),
         Category("Facebook Like"),
         DefaultValue(false),
         Browsable(true),
         Bindable(true)]
        public bool Compact
        {
            get { return compact; }
            set
            {
                if (compact == value) return;
                compact = value;
                updateBtn();
            }
        }

        /// <summary>
        /// The action of the button
        /// </summary>
        [Description("Verb"),
         Category("Facebook Like"),
         DefaultValue(Verbs.Like),
         Browsable(true),
         Bindable(true)]
        public Verbs Verb
        {
            get { return verb; }
            set
            {
                if (verb == value) return;
                verb = value;
                updateBtn();
            }
        }

        /// <summary>
        /// Use a compact button layout
        /// </summary>
        [Description("Verb"),
         Category("Facebook Like"),
         DefaultValue(ColorSchemes.Light),
         Browsable(true),
         Bindable(true)]
        public ColorSchemes ColorScheme
        {
            get { return colorscheme; }
            set
            {
                if (colorscheme == value) return;
                colorscheme = value;
                updateBtn();
            }
        }

        public FacebookLikeButton()
        {
             InitializeComponent();
        }

        private void webBrowser_DocumentCompleted(object sender, WebBrowserDocumentCompletedEventArgs e)
        {
            WebBrowser wb = (WebBrowser)sender;

            // check if the actual page was loaded, cancel on error
            if (wb.Document.Url.Port == -1) return;

            // apply background
            Color bg = this.BackColor;
            StringBuilder sb = new StringBuilder("background-color:rgb(");
            sb.Append(bg.R); sb.Append(',');
            sb.Append(bg.G); sb.Append(',');
            sb.Append(bg.B); sb.Append(");");
            wb.Document.Body.Style = sb.ToString();
            wb.Document.BackColor = bg;

            webBrowser.Show();
        }

        private void FacebookLikeButton_Load(object sender, EventArgs e)
        {
            updateBtn();
        }

        static Image bgImage = null;

        protected override void OnPaintBackground(PaintEventArgs e)
        {
            base.OnPaintBackground(e);

            if (this.DesignMode)
            {
                if (bgImage == null)
                    bgImage = ToolboxBitmapAttribute.GetImageFromResource(typeof(FacebookLikeButton), "Facebook16.bmp", false);

                Point p = (Point)(this.Size - bgImage.Size);
                p.X /= 2; p.Y /= 2;
                e.Graphics.DrawImage(bgImage, p);
            }
        }
    }
}
